<?php
class GDMS {

	private $username;
	private $password;
	private $apiID;
	private $secretKey;

	private $accessToken;
	private $debug;

	private $domain;
	private $version;

	public function __construct(int $apiID, String $secretKey, String $username, String $password, bool $eu = false) {
		$this->domain = $eu ? "eu.gdms.cloud" : "www.gdms.cloud";
		$this->version = '1.0.0';
		$this->username = $username;
		$this->password = $password;
		$this->apiID = $apiID;
		$this->secretKey = $secretKey;
		$this->accessToken = null;
		$this->debug = false;
	}

	public function setDebug(bool $enabled) {
		$this->debug = $enabled;
	}

	private function debug(String $str) {
		if ($this->debug) {
			Yii::log($str, CLogger::LEVEL_INFO, 'application.GDMS');
		}
	}

	private static function get(String $url) {
		$ch = curl_init();
		$options = array(
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_SSL_VERIFYPEER => false,
			CURLOPT_HTTPAUTH => CURLAUTH_DIGEST,
			CURLOPT_SSL_VERIFYHOST => false,
			CURLOPT_URL => $url,
			CURLOPT_HTTPHEADER => array("Content-Type: application/json")
		);
		curl_setopt_array($ch, $options);
		$response = curl_exec($ch);
		if (curl_error($ch)) {
			echo curl_error($ch);
			curl_close($ch);
		} else {
			curl_close($ch);
			$response = json_decode($response, true);
			return $response;
		}
		return NULL;
	}

	private static function timestamp() : int {
		return round(microtime(true) * 1000);
	}

	private function signature(int $timestamp, String $body = "") : String {
		$str = "&access_token=" . $this->accessToken . "&client_id=" . $this->apiID . "&client_secret=" . $this->secretKey . "&timestamp=" . $timestamp . "&";
		if (strlen($body) > 0) {
			$bodyHash = hash("sha256", $body);
			$str = $str . $bodyHash . "&";
		}
		$hash = hash("sha256", $str);
        return $hash;
	}

	private function basicParams(String $body) : String {
		$timestamp = static::timestamp();
		$signature = $this->signature($timestamp, $body);
		return "access_token=" . $this->accessToken . "&signature=$signature&timestamp=$timestamp";
	}

	private static function post(String $endpoint, array $params = array()) {
		$json = !empty($params) ? json_encode($params) : null;
		$ch = curl_init();
		$options = array(
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_SSL_VERIFYPEER => false,
			CURLOPT_POST => true,
			CURLOPT_POSTFIELDS => $json,
			CURLOPT_URL => $endpoint,
			CURLOPT_HTTPHEADER => array("Content-Type: application/json")
		);
		curl_setopt_array($ch, $options);
		$response = curl_exec($ch);
		if (curl_error($ch)) {
			echo curl_error($ch);
			curl_close($ch);
		} else {
			curl_close($ch);
			$response = json_decode($response, true);
			return $response;
		}
		return NULL;
	}

	private function gsPost(String $endpoint, array $params = array()) {
		$body = !empty($params) ? json_encode($params) : '';
		$url = 'https://' . $this->domain . '/oapi/v' . $this->version . '/' . $endpoint . '?' . $this->basicParams($body);
		$this->debug("POST: $url\r\n$body");
		return static::post($url, $params);
	}

	private function gsGet0(String $endpoint, array $params) {
		$body = http_build_query($params);
		$url = 'https://' . $this->domain . '/oapi/' . $endpoint . '?' . $body;
		$this->debug("GET: $url");
		return static::get($url);
	}

	private function gsGet(String $endpoint, array $params = array()) {
		$body = http_build_query($params);
		$url = 'https://' . $this->domain . '/oapi/' . 'v' . $this->version . '/' . $endpoint . '?' . $this->basicParams($body);
		if (strlen($body) > 0) {
			$url .= "&" . $body;
		}
		$this->debug("GET: $url");
		return static::get($url);
	}

	public function login() {
        $params = array(
            'grant_type' => 'password',
            'password' => $this->password,
            'username' => $this->username,
            'client_id' => $this->apiID,
            'client_secret' => $this->secretKey
        );
    
        $res = $this->gsGet0('oauth/token', $params);
    
        if ($this->debug) {
            Yii::log("🔐 Login Response: " . json_encode($res), CLogger::LEVEL_INFO, 'application.GDMS');
        }
    
        if ($res && isset($res['access_token'])) {
            $this->accessToken = $res['access_token'];
            $this->debug("✅ Logged in successfully");
            return 0;
        }
    
        return -1;
    }

	public function organizationList() {
		return $this->gsGet('org/list');
	}

	public function siteList(int $orgID = 0) {
		$params = array();
		if ($orgID > 0) {
			$params['orgId'] = $orgID;
		}
		return $this->gsPost('site/list', $params);
	}

	public function deviceList() {
		return $this->gsPost('device/list');
	}

	public function getDevice(String $mac) {
		$params = array();
		$params['mac'] = $mac;
		return $this->gsPost('device/list', $params);
	}

	public function addDevice(String $mac, String $sn, int $siteID, String $deviceName = '', int $orgID = 0) {
		$params = array();
		$params[0] = array(
			'mac' => $mac,
			'sn' => $sn,
			'siteId' => $siteID
		);
		if (strlen($deviceName) > 0) {
			$params[0]['deviceName'] = $deviceName;
		}
		if ($orgID > 0) {
			$params[0]['orgId'] = $orgID;
		}
		return $this->gsPost('device/add', $params);
	}

	public function editDevice(String $mac, String $sn, int $siteID, String $deviceName = '', int $orgID = 0) {
		$params = array(
			'mac' => $mac,
			'sn' => $sn,
			'siteId' => $siteID
		);
		if (strlen($deviceName) > 0) {
			$params['deviceName'] = $deviceName;
		}
		if ($orgID > 0) {
			$params['orgId'] = $orgID;
		}
		return $this->gsPost('device/edit', $params);
	}

	public function deviceDetails(String $mac, bool $isFirst) {
		$params = array(
			'mac' => $mac,
			'isFirst' => $isFirst ? 1 : 0
		);
		return $this->gsPost('device/detail', $params);
	}

	public function deviceAccountStatus(String $mac) {
		$params = array('mac' => $mac);
		return $this->gsPost('device/account/status', $params);
	}

	public function deviceAccountConfig(String $mac) {
		$params = array('mac' => $mac);
		return $this->gsPost('device/account/info', $params);
	}

	private function addTask(String $taskName, int $taskType, String $mac, int $execType, String $fwDownloadURL = '', int $orgID = 0) {
		$params = array(
			'taskName' => $taskName,
			'taskType' => $taskType,
			'macList' => array($mac),
			'execType' => $execType
		);
		if (strlen($fwDownloadURL) > 0) {
			$params['firmwareDownloadUrl'] = $fwDownloadURL;
		}
		if ($orgID > 0) {
			$params['orgId'] = $orgID;
		}
		return $this->gsPost('task/add', $params);
	}

	public function deviceReboot(String $mac, int $orgID = 0) {
		$uniqueTaskName = static::timestamp() . "_" . $mac . "_Reboot";
		return $this->addTask($uniqueTaskName, 1, $mac, 1, '', $orgID);
	}

	public function deviceFactoryReset(String $mac, int $orgID = 0) {
		$uniqueTaskName = static::timestamp() . "_" . $mac . "_Reset";
		return $this->addTask($uniqueTaskName, 2, $mac, 1, '', $orgID);
	}
}
?>